import numpy as np 
import matplotlib.pyplot as plt

# Runs the trajectory of a single Brownian particle, plots it, and computes the MSD

# fixed parameters
zeta = 1.0 # friction coefficient
kT = 0.01 # temperature
dt = 0.1 # time step (needs to be small enough). 
# Here this means that all three positon and angle updates 1) v0 dt, 2) \sqrt(2 kT/zeta dt) and 3) \sqrt(2 Dr dt) are all much below the particle radius (=1) / \pi/2. 
# Erring on the side of caution as numerical efficiency is not a major issue here

# we want to vary v0 and Dr, hence put the integration into a function that we can call
# optional 3rd argument: number of simulation steps, make longer for small Dr
def active_brownian(v0,Dr, Nsteps = 2000):
    # create time array for ease of plotting later-on
    tval = np.linspace(0,Nsteps*dt,Nsteps)
    # create empty arrays for the x, y and angle coordinates
    # implicitly inital condition x=y=theta=0
    xval = np.zeros((Nsteps,))
    yval = np.zeros((Nsteps,))
    theta = np.zeros((Nsteps,))
    # compute the noise prefactors once and for all for efficiency
    # np.random.randn() is Gaussian random number with mean 0 and width 1
    noiseamp_xy = np.sqrt(2*kT/zeta*dt)
    noiseamp_theta = np.sqrt(2*Dr*dt)
    # main loop update including Euler-Maruyama integrator
    for k in range(Nsteps-1):
        xval[k+1] = xval[k] + np.cos(theta[k])*dt + noiseamp_xy*np.random.randn() 
        yval[k+1] = yval[k] + np.sin(theta[k])*dt + noiseamp_xy*np.random.randn() 
        theta[k+1] = theta[k] + noiseamp_theta*np.random.randn() 
    
    # return trajectory data
    return tval,xval,yval,theta

# We now compute mean square displacements for angle and position. 
# For this, we will use a function as well to pass it one trajectory at a time
def calc_MSD(x,y,angle):
    Npts = len(x)
    msd_theta = np.zeros((Npts,))
    msd_pos = np.zeros((Npts,))
    # Compute MSDs at timestep k over all possible intervals so that both 
    # k0 and k0 + k are within the 0 to Npts of the simulation
    # sliding (time) average that improves statistics
    for k in range(Npts):
        msd_theta[k] = np.average((angle[k:]-angle[:(Npts-k)])**2) 
        msd_pos[k] = np.average((x[k:]-x[:(Npts-k)])**2 + (y[k:]-y[:(Npts-k)])**2)
    # return both msd results
    return msd_theta, msd_pos

# Theoretical MSD predictions
def theo_MSD(v0,Dr,Dt,tval):
    msd_theta = 2*Dr*tval
    msd_pos = 4*Dt*tval + 2*v0**2/Dr*(tval - 1.0/Dr*(1- np.exp(-Dr*tval)))
    return msd_theta, msd_pos


# To start with, a couple of plots
# 1. Passive limit: v0 = 0, Dr = 1 (value irrelevant)
tval,xval,yval,theta = active_brownian(0.0,1.0)
# Trajectory plot
plt.figure()
plt.plot(xval,yval)
plt.axis('equal')
plt.xlabel('x')
plt.ylabel('y')
plt.title('Passive trajectory')

# 2. Active with v0 = 1 and progressively smaller rotational diffusion
# As one can see, quite some thermal fluctuations in all of them due to the translational component
plt.figure()
Drval = [1.0,0.1,0.01]
v0=1.0
cols = ['r','g','b','m','y']
u=0
for Dr in Drval:
    tval,xval,yval,theta = active_brownian(v0,Dr) # v0 = 1
    plt.plot(xval,yval,color=cols[u],label='Dr='+str(Dr))
    u+=1
plt.axis('equal')
plt.xlabel('x')
plt.ylabel('y')
plt.legend()
plt.title('Trajectory for v0=' + str(v0))

# 3. At the lowest Dr = 0.01, if we increase v0 we see the persistent random walk more clearly
plt.figure()
v0val = [0.1,1.0,10.0]
Dr=0.01
cols = ['r','g','b','m','y']
u=0
for v0 in v0val:
    tval,xval,yval,theta = active_brownian(v0,Dr) # Dr = 0.01
    plt.plot(xval,yval,color=cols[u],label='v0='+str(v0))
    u+=1
plt.axis('equal')
plt.xlabel('x')
plt.ylabel('y')
plt.legend()
plt.title('Trajectory for Dr=' + str(Dr))

# Now the MSDs, see function calc_MSD. For Illustration, we will vary Dr first
Drval = [0.01,0.03,0.1,0.3,1.0]
v0 = 1.0
# switch of which figure to plot (angle or )
plot_space = True
# Groundwork for the plot
plt.figure()
cols = ['r','g','b','m','y']
u=0
for Dr in Drval:
    print(Dr)
    tval,xval,yval,theta = active_brownian(1.0,Dr,10000) # v0 = 1
    # simulation calculation
    msd_theta, msd_pos = calc_MSD(xval,yval,theta)
    # theoretical preditions
    theo_msd_theta, theo_msd_pos = theo_MSD(v0,Dr,kT/zeta,tval)
    if plot_space:
        plt.loglog(tval[1:],msd_pos[1:],lw=2,color=cols[u],label='Dr='+str(Dr))
        plt.loglog(tval[1:],theo_msd_pos[1:],':',lw=2,color=cols[u])
    else:
        plt.loglog(tval[1:],msd_theta[1:],lw=2,color=cols[u],label='Dr='+str(Dr))
        plt.loglog(tval[1:],theo_msd_theta[1:],':',lw=2,color=cols[u])
    u+=1
plt.xlabel('time')
if plot_space:
    plt.ylabel('spatial MSD')
    plt.legend()
    plt.title('Spatial MSD for v0 = ' +str(v0))
else:
    plt.ylabel('angular MSD')
    plt.legend()
    plt.title('Angular MSD for v0 = ' +str(v0))



plt.show()
